<?php

namespace App\Http\Controllers;

use App\Models\Label;
use App\Http\Requests\StoreLabelRequest;
use App\Http\Requests\UpdateLabelRequest;

use App\Models\Sublabel;
use App\Http\Requests\StoreSublabelRequest;
use App\Http\Requests\UpdateSublabelRequest;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Models\User;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Arr;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Session;

class LabelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function __construct(){
        $this->middleware('auth');
        $this->middleware('permission:create-label|edit-label|delete-label', ['only' => ['index', 'show']]);
        $this->middleware('permission:create-label', ['only' => ['create', 'store']]);
        $this->middleware('permission:edit-label', ['only' => ['edit', 'update']]);
        $this->middleware('permission:delete-label', ['only' => ['destroy']]);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $roleId = Role::LABEL; 
        $userIds  = [];
        $user_roles = DB::table('role_user')->select('user_id')->where('role_id',$roleId)->get();
        foreach($user_roles as $user_role){
            $userIds[] = $user_role->user_id;
        }
        if(!empty($userIds)){
            $users  = User::whereIn('id',$userIds)->latest('id')->paginate(itemsPerPage());
        }else{
            $users  = User::where('id','test')->latest('id')->paginate(itemsPerPage());
        }

        // Access related users  
        return view('users.index', [
            'users' => $users,
            'userFlag' => true,
            'user_roles' => 'Label'
        ]);

    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        user_logs();
        $roles = Role::pluck('name', 'name')->all();
        $countries = Country::get();
        $userFlag = true;
        $user_roles = 'Label';
        return view('users.create', compact('roles', 'countries', 'userFlag','user_roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreUserRequest $request)
    {
        $input = $request->all();
        $input['password'] = Hash::make($request->password);
        $user = User::create($input);
        $user->assignRole($request->roles);

        return redirect()->route('label.index')->with([
            'message' => 'New sublabel is added successfully.',
            'alert-type' => 'success'
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return redirect()->route('users.index');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user,$id)
    {
        // echo $id;die();
        if(!empty($id)){
            $user = User::find($id);
        }
        // Check Only Super Admin can update his own Profile
        if ($user->hasRole('Super Admin')) {
            if ($user->id != auth()->user()->id) {
                abort(403, 'USER DOES NOT HAVE THE RIGHT PERMISSIONS');
            }
        }
        return view('users.edit', [
            'user' => $user,
            'userFlag' => true,
            'user_roles' => 'Label',
            'roles' => Role::pluck('name')->all(),
            'userRoles' => $user->roles->pluck('name')->all()
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateUserRequest $request, User $user)
    {

        $input = $request->all();
        if (!empty($request->password)) {
            $input['password'] = Hash::make($request->password);
        } else {
            $input = $request->except('password');
        }
        $user->update($input);
        $user->syncRoles($request->roles);

        return redirect()->route('label.index')->with([
            'message' => 'Label is updated successfully.',
            'alert-type' => 'success'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // About if user is Super Admin or User ID belongs to Auth User
        if ($user->hasRole('Super Admin') || $user->id == auth()->user()->id) {
            abort(403, 'USER DOES NOT HAVE THE RIGHT PERMISSIONS');
        }

        $user->syncRoles([]);
        $user->delete();
        // return redirect()->route('users.index')
        //     ->withSuccess('User is deleted successfully.');

        return redirect()->route('label.index')->with([
            'message' => 'Label is deleted successfully.',
            'alert-type' => 'success'
        ]);
    }
}
